<?php
/**
 * PROV plugin for Dokuwiki
 * 
 * This plugin stores PROV data in turtle notation files every time a page is created,
 * modified or deleted.
 *
 * @license GPL 2 http://www.gnu.org/licenses/gpl-2.0.html
 * @author  Piotr Mitana <the.mail.of.mi2@gmail.com>, Krzysztof Kutt <kk@agh.edu.pl>
 */

// Ensure the script is run on Dokuwiki
if(!defined('DOKU_INC')) die();

/**
 * This class represents a plugin instance during single dokuwiki request procession.
 */
class action_plugin_prov_prov extends DokuWiki_Action_Plugin
{
	private $page = false;  // Page ID, including namespaces
	private $previousRev = false;  // Last revision
	private $currentRev = false;  // Current revision (probably equal to current unix time)
	private $username = false;
	private $links = false;  // Array of URL-s used in this site change
	private $comment = false;  // Comment entered by user
	private $operation = false;  // 'create', 'edit', 'delete' or empty before the operation type is determined
	private $wikiUrl = false;  // URL of this wiki based on the $_SERVER values
	
	/**
	 * This function generates PROV prefixed for the Wiki address specified.
	 * It is called witch $this->wikiUrl after the latter one is calculated.
	 */
	private function provPrefixes($adres_wiki)
	{
		return "@prefix prov: <http://www.w3.org/ns/prov#> .\n" .
				"@prefix dc: <http://purl.org/dc/elements/1.1/> .\n" .
				"@prefix lokipage: <{$adres_wiki}> .\n" .
				"@prefix lokievent: <{$adres_wiki}special:lokievent#> .\n" .
				"@prefix lokiuser: <{$adres_wiki}user:> .";
	}

	/**
	 * This function generates turtle data for the page creation (activated the page content has been empty before the current change)
	 */
	private function provCreation($strona, $nowaRev, $login, $komentarz, $linki)
	{
		$autor = ($login != "") ? "lokiuser:$login" : "lokiuser:special:guest";  // Ensure the right author if the change is performed by the quest
		
		return "lokipage:{$strona} a prov:eEntity .\n" .
			"lokipage:{$strona}.{$nowaRev} a prov:eEntity ;\n" .
			"\tprov:specializationOf lokipage:{$strona} ;\n" .
			"\tprov:wasGeneratedBy lokievent:created_{$strona}.{$nowaRev} .\n" .
			"lokievent:created_{$strona}.{$nowaRev} a prov:Activity ;\n" .
			"\tprov:wasAssociatedWith {$autor} ;\n" .
			"\tlokirel:comment \"{$komentarz}\"" . 
			((count($linki) > 0) ?
				(" ;\n\tprov:used " . implode(" ,\n\t\t", $linki) . " .\n\n") :
				(" .\n\n"));
	}
	
	/**
	 * This function generates turtle data for the page modification (the page content neither was nor will be empty)
	 */
	private function provEdition($strona, $nowaRev, $staraRev, $login, $komentarz, $linki)
	{
		$linki = array_merge(array("lokipage:{$strona}.{$staraRev}"), $linki);  // Prepend the previous revision to the used sites
		$autor = ($login != "") ? "lokiuser:$login" : "lokiuser:special:guest";  // Ensure the right author if the change is performed by the quest
		
		return "lokipage:{$strona}.{$nowaRev} a prov:Entity ;\n" .
			"\tprov:specializationOf lokipage:{$strona} ;\n" .
			"\tprov:wasRevisionOf lokipage:{$strona}.{$staraRev} ;\n" .
			"\tprov:wasGeneratedBy lokievent:edited_{$strona}.{$nowaRev} .\n" .
			"lokievent:edited_{$strona}.{$nowaRev} a prov:Activity ;\n" .
			"\tprov:wasAssociatedWith {$autor} ;\n" .
			"\tlokirel:comment \"{$komentarz}\" ;\n" .
			"\tprov:used " . implode(" ,\n\t\t", $linki) . " .\n\n";
	}
	
	/**
	 * This function generates turtle data for the page deletion (the empty content is to be set for the page)
	 */
	private function provDeletion($strona, $nowaRev, $staraRev, $login, $komentarz)
	{
		$autor = ($login != "") ? "lokiuser:$login" : "lokiuser:special:guest";

		return "lokievent:deleted_{$strona}.{$nowaRev} a prov:Activity ;\n" .
			"\tprov:used lokipage:{$strona}.{$staraRev} ;\n" .
			"\tprov:wasAssociatedWith {$autor} ;\n" .
			"\tlokirel:comment \"{$komentarz}\" .\n\n";
	}
	
	/**
	 * This is a helper function thaa returns the list of all pages (more scrictly - their ID's) available in the wiki.
	 */
	private function listPages($dir = "")
	{
		$path = "data/pages/$dir";
		$res = "";
		
		foreach(scandir($path) as $file)
		{
			if($file == "." || $file == "..")
				continue;
			
			else if(preg_match("/^.*\.txt$/", $file))
			{
				if($dir != "") $res .= substr("$dir/$file", 1, -4) . "\n";
				else $res .= substr($file, 0, -4) . "\n";
			}
			
			else if(is_dir("$path/$file"))
				$res .= $this->listPages("$dir/$file") . "\n";
		}
		
		$res = str_replace("/", ":", $res);
		$res = str_replace("\\", ":", $res);
		$res = str_replace("\\", ":", $res);
		return substr($res, 0, -1);
	}

	/**
	 * A function invoked by DokuWiki during plugin loading. It registers all the event handlers.
	 */
	public function register(Doku_Event_Handler $controller)
	{
		$controller->register_hook('HTML_EDITFORM_OUTPUT', 'BEFORE', $this, 'handle_html_editform_output');
		$controller->register_hook('IO_WIKIPAGE_WRITE', 'BEFORE', $this, 'handle_io_wikipage_write');
	}

	/**
	 * An event handler, which injects additional fields into the site edit form. The additional
	 * fields allow user specify the sites used during the current change. They can be entered by URL or
	 * selected from the list of all wiki pages.
	 */
	public function handle_html_editform_output(Doku_Event &$event, $param)
	{
		// Get the "submit" button position
		$pos = $event->data->findElementByAttribute('type', 'submit');
		
		// If there's no "submit" button, it can't be an edit form.
		if(!$pos)
			return;
		
		// Insert some Javascript to fill in the wiki's page list and add it's filterability.
		$html = '<script>
			function pageAdded()
			{
				var usedsites = document.getElementById("usedsites");
				var list = document.getElementById("pagelist");
				var page = list.options[list.selectedIndex].value;
				
				if(usedsites.value != "" && !usedsites.value.endsWith("\n"))
					usedsites.value += ("\nlokipage:" + page + "\n");
				
				else
					usedsites.value += ("lokipage:" + page + "\n");
			}
			function filterChanged()
			{
				var pages = document.getElementById("avpages").value.split("\n");
				var list = document.getElementById("pagelist");
				var filter = document.getElementById("pagefilter").value;
				
				list.innerHTML = "";  // Usuń wszystkie opcje
				
				for(var i = 0; i < pages.length; i++)
				{
					if(pages[i].indexOf(filter) >= 0)
						list.innerHTML += "<option value=\'" + pages[i] + "\'>" + pages[i] + "</option>";
				}
			}
		</script>';
		
		// Break line and begin custom section
		$html .= '
			<br/>
			<div>
				<!-- <h4>PROV</h4> -->';
		
		// Textarea for lins with label
		$html .= '
				<div style="display: inline-block; vertical-align: top;">
				<label style="cursor: text;">URIs/URLs for used resources (put one URI in one line):</label><br/>
				<textarea name="usedSites" id="usedsites" cols="60" rows="6"></textarea>
			</div>';
		
		// List for the available pages (empty so far) with label
		$html .= '
			<div style="display: inline-block; vertical-align: top; margin-left: 10px;">
				<label style="cursor: text;">Select wiki pages that were used as resources (double-click on page):</label></br>
				<select id="pagelist" size="5" style="width: 350px;" ondblclick="pageAdded();"></select></br>';
		
		// Filter text field with label. On key release the list is updated
		$html .= '
			<label style="cursor: text;">Filter: </label>
			<input type="text" id="pagefilter" onkeyup="filterChanged();" size="35"/>';
		
		// Hidden field containing the list of all pages. It is used by filtering function.
		$html .= '<input type="hidden" id="avpages" value="' . $this->listPages() . '"/>';
		
		// Instantly fire empty filter to show all the sites in the list.
		$html .= '<script>filterChanged();</script>';
		$html .= '</div>';
		
		// End the custom section
		$html .= '</div><br/>';
		
		// Inject the code into the form.
		$event->data->insertElement($pos, $html);
	}

	/**
	 * An event handler which performs the PROV file update. May be triggered two times - once when the previous revision
	 * is moved into the attic and once again when the new revision is created.
	 */
	public function handle_io_wikipage_write(Doku_Event &$event, $param)
	{
		// If the namespace path is present, prepend it to the page name to get the full ID.
		$this->page = ($event->data[1] !== false ? ($event->data[1] . ":") : "") . $event->data[2];
		
		/* Grab the previous revision by checking the page file modification time. Plese note that the
		   current change has not yet been performed. If the file does not exists, the previous revision does not exist. */
		$this->previousRev = filemtime($event->data[0][0]) !== false ? filemtime($event->data[0][0]) : $this->previousRev;
		
		/* Grab the current revision from the event. This one is empty when the moving to attic is signaled.
		   Leave it unchanged then, will be set later. */
		$this->currentRev = $event->data[3] !== false ? $event->data[3] : $this->currentRev;
		
		// Some other variables
		$this->username = pageinfo()['userinfo']['name'];
		$this->links = array_unique(array_filter(explode("\n", str_replace("\r", "", $_POST['usedSites']))));  // Filter out duplcates & empty strings
		$this->comment = $_POST['summary'];
		
		// Determine the wiki addres based on the $_SERVER values
		$url = "http://$_SERVER[HTTP_HOST]$_SERVER[REQUEST_URI]";
		$hasGetList = false;
		
		// If URL contains the $_GET data, ditch it
		$qmPosition = strpos($url, "?");
		if($qmPosition !== false)
		{
			$hasGetList = true;
			$url = substr($url, 0, $qmPosition);
		}
		
		// If URL ends witch the page ID, remove it and the rest is the wiki addres.
		if(strrpos($url, $this->page) == strlen($url) - strlen($this->page))
			$this->wikiUrl = substr($url, 0, strlen($url) - strlen($this->page));
		
		// If not, it's probably doku.php?id=page.
		else if($hasGetList)
			$this->wikiUrl = $url . "?id=";
		
		/* The $this->operation variable is set near the of the first handler call. The operations are performed in:
		   - "create" and "edit" - 1st call,
		   - "delete" - 2nd call.
		   Therefore if the opreation is set, it's the 2nd call. Unless the opration is "delete", it's already done. */
		if($this->operation != false && $this->operation != 'delete')
			return;
		
		// If operation is not set, determine it based on revisions and content.
		if($this->operation == false && $this->previousRev == false && $this->currentRev != false)  // No previous revision
			$this->operation = 'create';
		
		else if($this->operation == false && $event->data[0][1] === '' && $event->data[0][2] === false)  // Empty content
			$this->operation = 'delete';
		
		else if($this->operation == false && $this->currentRev != false && $this->previousRev != false)  // Both previous and current revisions
			$this->operation = 'edit';
		
		// If operation is not set, return.
		if($this->operation == false)
			return;
		
		// Get the prov directory and create it if needed.
		$dir = 'data/prov/' . ($event->data[1] !== false ? (str_replace(":", "/", $event->data[1]) . "/") : "");
		$filename = $event->data[2] . '.ttl';
		$rev = $event->data[3];
		
		if(!file_exists($dir)) mkdir($dir);
		
		// If file does not exist, prefixes will be stored at its beginning.
		$needsPrefixes = !file_exists("$dir$filename");
		
		// Wrap all the external URLs into angular brackets
		for($i = 0; $i < count($this->links); $i++)
		{
			$pos = strpos($this->links[$i], "lokipage:");
			if($pos === false || $pos > 0)
				$this->links[$i] = "<{$this->links[$i]}>";
		}
		
		// Get the turtle notation data
		if($this->operation == 'create')
			$prov = $this->provCreation($this->page, $this->currentRev, $this->username, $this->comment, $this->links);
		
		if($this->operation == 'edit')
			$prov = $this->provEdition($this->page, $this->currentRev, $this->previousRev, $this->username, $this->comment, $this->links);
		
		if($this->operation == 'delete' && $this->currentRev != false && $this->previousRev != false)  // Entered in the 2nd call
			$prov = $this->provDeletion($this->page, $this->currentRev, $this->previousRev, $this->username, $this->comment);
		
		// Store it to file
		$turtlefile = fopen("$dir$filename", "a");
		
		if($needsPrefixes)
			fwrite($turtlefile, $this->provPrefixes($this->wikiUrl));
		
		fwrite($turtlefile, $prov);
		fclose($turtlefile);
	}
}
