;(function($) {

	$.fn.extend({
		BPMNDiagram: function(options) {
		
			var settings = {
				auto: true,
				diagramClass: 'bpmn-diagram',
				diagram: null, //'after','before'
				hideCode: true,
				subprc: null, //'collapse', 'expand'
				debug: false
			};
			$.extend(settings,options);
			
			if( settings.diagram === null ) {
				settings.diagram = $(this).data('bpmn-diagram');
			}
			
			if( settings.subprc === null ) {
				settings.subprc = $(this).data('bpmn-subprc');
			}
		
			return $(this).each(function(){
				$(this).data('diagram',new Diagram(this,settings));
			});
		
		}
	});

	/* Diagram */
	var Diagram = Base.extend({
		constructor: function(codeNode,settings){
			this.codeNode = $(codeNode);
			this.settings = settings;
		
			if( this.settings.auto ) {
				this.generate();
			}
		},
		
		_isSimple: function(code){
			return !code.activities;
		},
	
		generate: function(){
			var diagram = this;
			this.destroy();
			this._createDOM();
			
			setTimeout(function(){
				diagram._loadCode(function(){
		
					//set drawing font
					this.context.font = this.diagramNode.css('font');
		
					var id = this.codeNode.attr('id') ? this.codeNode.attr('id')+'-diagram' : 'diagram' + (new Date().getTime() );
			
					//if it is a task (not process/subprocess) then workaround a bit
					if( this._isSimple(this.code) ) {
						this.code = { activities: { singular: $.extend({},this.code) } };
					}
			
					this.ge = new GroupingElement(this.code,id,null,this.context);
					this.ge._calculatePositions();
		
					var bb = this.ge.getBoundingBox(true);
					var canvas = this.diagramNode.find('canvas').get(0);
					canvas.width = Math.max(bb[2],this.diagramNode.width());
					canvas.height = Math.max(bb[3],this.diagramNode.height());
		
					//set background to white
					this.context.save();
					this.context.fillStyle = '#fff';
					this.context.fillRect(0,0,canvas.width,canvas.height);
					this.context.restore();
		
					//set drawing font (again, since changing canvas size resets the context)
					this.context.font = this.diagramNode.css('font');
					this.context.textBaseline = 'top';
					this.context.textAlign = 'center';
					this.ge.draw(this.settings.debug);
					
					//trigger complete signal
					diagram.codeNode.trigger('diagramGenerated');
				});
			},50);
		},
	
		destroy: function(){
			if( this.diagramNode ) {
				this.diagramNode.remove();
			}
		},
	
		_createMenu: function(){
			this.menuNode = $('<div class="'+this.settings.diagramClass+'-menu"></div>').appendTo(this.diagramNode);
		
			var zoomNormal = $('<a href="#" class="zoom-normal" title="100%"><img src="'+BPMN_IMG_PATH+'zoom-normal.png" width="48" height="48" alt="100%" /></a>').appendTo(this.menuNode).hide();
			var zoomIn = $('<a href="#" class="zoom-in" title="Zoom In"><img src="'+BPMN_IMG_PATH+'zoom-in.png" width="48" height="48" alt="Zoom In" /></a>').appendTo(this.menuNode);
			var zoomOut = $('<a href="#" class="zoom-out" title="Zoom Out"><img src="'+BPMN_IMG_PATH+'zoom-out.png" width="48" height="48" alt="Zoom Out" /></a>').appendTo(this.menuNode);
		
			var viewFullscreen = $('<a href="#" class="view-fullscreen" title="View Fullscreen"><img src="'+BPMN_IMG_PATH+'view-fullscreen.png" width="48" height="48" alt="View Fullscreen" /></a>').appendTo(this.menuNode);
			var closeFullscreen = $('<a href="#" class="close-fullscreen" title="Close Fullscreen"><img src="'+BPMN_IMG_PATH+'close-fullscreen.png" width="48" height="48" alt="Close Fullscreen" /></a>').appendTo(this.menuNode).hide();
		
			var save = $('<a href="#" class="save" title="Save as PNG"><img src="'+BPMN_IMG_PATH+'document-save.png" width="48" height="48" alt="Save" /></a>').appendTo(this.menuNode);
		
			var zoom = function(canvas,dn,scale,nw) {
				canvas.width(nw || canvas.width()*scale);
				canvas.parent().scrollLeft(-canvas.position().left*scale-dn.width()*0.5+dn.width()*0.5*scale).scrollTop(-canvas.position().top*scale-dn.height()*0.5+dn.height()*0.5*scale);
			};
		
			zoomNormal.bind('click.bpmn',{ diagramNode: this.diagramNode, codeNode: this.codeNode },function(ev){
				ev.preventDefault();
				var dn = ev.data.diagramNode;
				var canvas = dn.find('canvas');
				var scale = canvas.get(0).width/canvas.width();
				zoom(canvas,dn,scale,canvas.get(0).width);
				$(this).hide().siblings('.zoom-in,.zoom-out').removeClass('disabled').filter('.zoom-out').hide();
			});
			zoomIn.bind('click.bpmn',{ diagramNode: this.diagramNode, codeNode: this.codeNode },function(ev){
				ev.preventDefault();
				if( !$(this).hasClass('disabled') ) {
					var dn = ev.data.diagramNode;
					var canvas = dn.find('canvas');
					var scale = Math.min(1.44,canvas.get(0).width*4/canvas.width());
					zoom(canvas,dn,scale);
					$(this).siblings('.zoom-normal').show().siblings('.zoom-out').removeClass('disabled').show();
					if( scale < 1.44 ) {
						$(this).addClass('disabled').hide();
					}
				}
			});
			zoomOut.bind('click.bpmn',{ diagramNode: this.diagramNode, codeNode: this.codeNode },function(ev){
				ev.preventDefault();
				if( !$(this).hasClass('disabled') ) {
					var dn = ev.data.diagramNode;
					var canvas = dn.find('canvas');
					var scale = Math.min(1.44,canvas.width()/dn.width());
					zoom(canvas,dn,1/scale);
					$(this).siblings('.zoom-normal').show().siblings('.zoom-in').removeClass('disabled').show();
					if( scale < 1.44 ) {
						$(this).addClass('disabled').hide();
					}
				}
			});
		
			viewFullscreen.bind('click.bpmn',{ diagram: this },function(ev){
				ev.preventDefault();
				$(this).hide().siblings('.close-fullscreen').show();
				ev.data.diagram.diagramNode.addClass('fullscreen').appendTo('body');
				$('body, html').css({
					'height': '100%',
					'overflow': 'hidden'
				});
			});
			
			closeFullscreen.bind('click.bpmn',{ diagram: this },function(ev){
				ev.preventDefault();
				$(this).hide().siblings('.view-fullscreen').show();
				ev.data.diagram.diagramNode.removeClass('fullscreen');
				ev.data.diagram._insertDiagramIntoDOM(ev.data.diagram);
				$('body, html').css({
					'height': '',
					'overflow': ''
				});
			});
		
			save.bind('click.bpmn',{ diagramNode: this.diagramNode, codeNode: this.codeNode },function(ev){
				ev.preventDefault();
				var canvas = ev.data.diagramNode.find('canvas').get(0);
				window.open(canvas.toDataURL(),"diagram.png");
			});
		},
		
		_insertDiagramIntoDOM: function(diagram){
			//is folded plugin used? if so, then we want diagram outside not inside...
			var element = diagram.codeNode.parent().hasClass('folded') ? ( diagram.settings.diagram == 'after' ? diagram.codeNode.parent() : diagram.codeNode.parent().prev() ) : diagram.codeNode;
		
			switch(diagram.settings.diagram) {
				case 'after':
					diagram.diagramNode.insertAfter(element);
					break;
				default:
					diagram.diagramNode.insertBefore(element);
					break;
			}
		},
		
		_normalizePoint: function(ev,element){
			var $e = $(element);
			var off = $e.offset();
			var x = ev.pageX - off.left;
			var y = ev.pageY - off.top;
			return [
				Math.round(x*(element.width/$e.width())),
				Math.round(y*(element.height/$e.height()))
			];
		},
	
		_createDOM: function(){
			var diagram = this;
			
			this.diagramNode = $('<div class="'+this.settings.diagramClass+'"></div>');
			this.diagramNodeImg = $('<div class="'+this.settings.diagramClass+'-image"></div>').appendTo(this.diagramNode);
			this.context = $('<canvas></canvas>').appendTo(this.diagramNodeImg).get(0).getContext("2d");
		
			this._insertDiagramIntoDOM(this);
			
			this.diagramNodeImg.bind('mousedown.bpmn',function(ev){
				if( ev.which === 1 ) {
					$(this).data('dragStart',[ev.pageX,ev.pageY]).bind('mousemove.bpmn', function(ev){
						var $t = $(this);
						var init = $t.data('dragCurrent') || $t.data('dragStart');
						if( init && Math.sqrt(Math.pow(init[0]-ev.pageX,2)+Math.pow(init[1]-ev.pageY,2)) > 5 ) {
							$t.addClass('drag');
							$t.scrollLeft($t.scrollLeft()+init[0]-ev.pageX);
							$t.scrollTop($t.scrollTop()+init[1]-ev.pageY);
							$t.data('dragCurrent',[ev.pageX,ev.pageY]);
						}
					});
				}
			}).bind('mouseup.bpmn mouseleave.bpmn',function(ev){
				$(this).data('dragStart',null).data('dragCurrent',null);
				$(this).unbind('mousemove.bpmn').removeClass('drag');
			}).bind('mousewheel.bpmn',function(ev,delta){
				var $t = $(this);
				var ct = $t.scrollTop();
				$t.scrollTop(ct-delta*72);
				if( ct != $t.scrollTop() ) {
					ev.preventDefault();
				}
			}).children('canvas').bind('mouseup.bpmn',function(ev){
					var dc = $(this).parent().data('dragCurrent');
					if( !dc && diagram.ge ) {
						var point = diagram._normalizePoint(ev,this);
						var act = diagram.ge.overClickableActivity(point);
						if( act ) {
							window.location = act.structure.href;
						}
					}
				}).bind('mousemove.bpmn',function(ev){
					if( diagram.ge ) {
						var point = diagram._normalizePoint(ev,this);
						var act = diagram.ge.overClickableActivity(point);
						if( act ) {
							diagram.diagramNode.css('cursor','pointer');
						} else {
							diagram.diagramNode.css('cursor','');
						}
					}
				});
		
			this._createMenu();
		
			if( this.settings.hideCode ) {
				this.codeNode.hide();
			}
		},
	
		_loadCode: function(codeReadyCallback){
			var cn = this.codeNode;
			
			if( cn.children('div').length ) {
				cn = cn.children('div').eq(0);
			}
			
			var code = cn.html();
			var i;
			var diagram = this;
			var resolved = [];
			
			var prepareCode = function(code){
				//normalize quotes
				code = code.replace(/[”“]/gim,'"');

				//remove comments
				code = JSON.minify(code);
		
				var fieldsRegExp = /(\{|,)\s*("|')?(\w+)\2?\s*:/gim;
				code = code.replace(fieldsRegExp,'$1 "$3":');

				var notQuotedRegExp = /:\s*(\w+)/gim;
				code = code.replace(notQuotedRegExp,': "$1"');

				var arrayRegExp = /:\s*\[([^\]]*)\]/gim;
				code = code.replace(arrayRegExp,function(m_full,m_1){
					var elements = m_1.match(/("|')?(\w+(\s+\w+)*)\1?/gim);
					var i;
					for(i = 0; i < elements.length; i++) {
						elements[i] = elements[i].replace(/("|')?(\w+(\s+\w+)*)\1?/gim,'"$2"');
					}
					return ': [ '+ elements.join(',') +' ]';
				});
			
				return code;
			};
			
			var doCompleted = function(){
				var j,completed,cn;
				
				completed = true;
				for(j = 0; j < resolved.length; j++) {
					completed = completed && resolved[j];
				}
				if (completed) {
					var cn = diagram.codeNode;
					if( cn.children('div').length ) {
						cn = cn.children('div').eq(0);
					}
					code = cn.html(code).text();
					code = prepareCode(code);
			
					diagram.code = $.parseJSON(code);
					
					//store info about links
					$links.each(function(index){
						diagram.code.activities[$(this).text()].href = $(this).attr('href');
					});
					
					if( diagram.settings.subprc === 'collapse' ){
						//collapse linked subprocesses
						$links.each(function(index){
							var act = diagram.code.activities[$(this).text()];
							if( !$.isEmptyObject(act.activities) ) {
								act.markers = act.markers || [];
								if( $.inArray("subprocess",act.markers) < 0 ) {
									act.markers.push("subprocess");
								}
							
								delete act.events;
								delete act.activities;
								delete act.gateways;
								delete act.flow;
							}
						});
					}
					
					codeReadyCallback.call(diagram);
				}
			};
			
			//resolve references to other pages
			var $links = $('<div />').html(code).find('a');
			
			//prepare for link validity checking
			//a "bit" roundabound, but much easier/safer than trying to get it right with regexp only
			var codeTmp = code;
			$links.each(function(index){
				codeTmp = codeTmp.replace($(this).get(0).outerHTML,$(this).text()+($(this).hasClass('bpmn-resolved') ? "" : ":{}"));
			});
			codeTmp = prepareCode(codeTmp);
			var jsonTmp = $.parseJSON(codeTmp);
			//leave only valid links in collection
			//namely those that are on first-level activities identificators
			$links = $links.filter(function(){
				return !!jsonTmp.activities[$(this).text()];
			});
			
			//are there any links left?
			if( $links.length ) {
				for(i = 0; i < $links.length; i++) {
					resolved[i] = false;
				}
			
				$links.each(function(index){
					var $a = $(this);
					var f = $a.get(0).outerHTML;
					
					//is this link already resolved (in backend)?
					if( $a.hasClass('bpmn-resolved') ) {
						
						resolved[index] = true;
						doCompleted();
						
					} else {
					
						setTimeout(function(){
							$.get($a.attr('href'),dataType="html").success(function(data){
								var c = $(data).find('.bpmn-code').text() || '{}';
							
								//match indents
								var m = code.match(new RegExp("(\\n(\\s)*)"+f));
								if( m ) {
									c = c.replace(/\n/gm,m[1]);
								}
					
								resolved[index] = true;
								code = code.replace(f,f+": "+c);
						
								doCompleted();
							});
						},50);
					
					}
				});
			} else {
				doCompleted();
			}
		}
	});

	/* Basic diagram elements */
	var Element = Base.extend({
		constructor: function(structure,id,group,context) {
			this.structure = structure;
			this.id = id;
			this.group = group;
			if( context ) {
				this.context = context;
			}
			this.offsetX = 0;
			this.offsetY = 0;
		},
		setOffsetX: function(value) {
			this.offsetX = value;
		},
		setOffsetY: function(value) {
			this.offsetY = value;
		},
		getBoundingBox: function(withAnnots){
			throw "Children need to re-implement getBoundingBox method!";
			//should return [offsetX, offsetY, width, height]
		},
		getPossibleEndpoints: function(start,connection,minDistance){
			var b = this.getBoundingBox();
			var p = [];
			if( start ) {
				p.push([ b[0]+b[2], b[1]+(b[3]/2), b[0]+b[2]+minDistance, b[1]+(b[3]/2) ]);
			} else {
				p.push([ b[0], b[1]+(b[3]/2), b[0]-minDistance, b[1]+(b[3]/2) ]);
			}
			p.push([ b[0]+(b[2]/2), b[1], b[0]+(b[2]/2), b[1]-minDistance ]);
			p.push([ b[0]+(b[2]/2), b[1]+b[3], b[0]+(b[2]/2), b[1]+b[3]+minDistance ]);
			return p;
		},
		getName: function(){
			var t = "";
			if( typeof(this.structure) === "string" ) {
				t = this.structure;
			} else if( this.structure.name ) {
				t = this.structure.name;
			}
			return t;  
		},
		draw: function(debug) {
			throw "Children need to re-implement draw method!";
		}
	});

	var ElementWithBoundary = Element.extend({

		getBoundaryEvents: function(){
			throw "Not implemented";
		},
	
		getPossibleEndpoints: function(start,connection,minDistance){
			if( typeof(connection[3]) != "undefined" ) {
				var el = this.getBoundaryEvents()[connection[3]];
				if( typeof(el) != "undefined" ) {
					var b = this.getBoundingBox();
					var bb = el.getBoundingBox();
					var point = [ b[0] + 20 + bb[0] + bb[2]/2, b[1] + 20 + bb[1]+bb[3], b[0] + 20 + bb[0] + bb[2]/2, b[1] + 20 + bb[1]+bb[3]+minDistance+40 ];
					return [ point ];
				}
			}
			return this.base(start,connection,minDistance);
		},
	
		_drawSubprocessMarker: function(x,y) {
			this.context.save();
		
			this.context.strokeRect(x+1.5,y+1.5,18,18);
		
			this.context.beginPath();
			this.context.moveTo(x+10.5,y+4.5);
			this.context.lineTo(x+10.5,y+16.5);
			this.context.moveTo(x+4.5,y+10.5);
			this.context.lineTo(x+16.5,y+10.5);
			this.context.stroke();
		
			this.context.restore();
		},
	
		_drawLoopMarker: function(x,y) {
			this.context.save();
		
			this.context.beginPath();
			this.context.arc(x+10,y+8,6,Math.PI/2,-1.25*Math.PI,true);
			this.context.lineTo(x+6.5,y+9.5);
			this.context.moveTo(x+2.6,y+11.5);
			this.context.lineTo(x+5.6,y+12.2);
			this.context.stroke();
		
			this.context.restore();
		},
	
		_drawParallelMarker: function(x,y){
			this.context.save();
		
			this.context.lineWidth = 3;
		
			this.context.beginPath();
			this.context.moveTo(x+4,y+2);
			this.context.lineTo(x+4,y+18);
			this.context.moveTo(x+10,y+2);
			this.context.lineTo(x+10,y+18);
			this.context.moveTo(x+16,y+2);
			this.context.lineTo(x+16,y+18);
		
			this.context.stroke();
		
			this.context.restore();
		},
	
		_drawSequentialMarker: function(x,y){
			this.context.save();
		
			this.context.lineWidth = 3;
		
			this.context.beginPath();
			this.context.moveTo(x+2,y+4);
			this.context.lineTo(x+18,y+4);
			this.context.moveTo(x+2,y+10);
			this.context.lineTo(x+18,y+10);
			this.context.moveTo(x+2,y+16);
			this.context.lineTo(x+18,y+16);
		
			this.context.stroke();
		
			this.context.restore();
		},
	
		_drawAdHocMarker: function(x,y){
			this.context.save();
		
			this.context.lineWidth = 2;
		
			this.context.beginPath();
			this.context.moveTo(x+3,y+12);
			this.context.bezierCurveTo(x+7,y,x+13,y+20,x+17,y+8);
		
			this.context.stroke();
		
			this.context.restore();
		},
	
		_drawCompensationMarker: function(x,y){
			this.context.save();
		
			this.context.beginPath();
		
			this.context.moveTo(x+3,y+10);
			this.context.lineTo(x+9,y+5.5);
			this.context.lineTo(x+9,y+14.5);
			this.context.lineTo(x+3,y+10);
			this.context.moveTo(x+9,y+10);
			this.context.lineTo(x+16,y+5.5);
			this.context.lineTo(x+16,y+14.5);
			this.context.lineTo(x+9,y+10);
		
			this.context.closePath();
		
			this.context.stroke();
		
			this.context.restore();
		},
	
		_drawMarker: function(x,y,type) {
			switch(type){
				case "subprocess":
					this._drawSubprocessMarker(x,y);
					break;
				case "loop":
					this._drawLoopMarker(x,y);
					break;
				case "parallel":
					this._drawParallelMarker(x,y);
					break;
				case "sequential":
					this._drawSequentialMarker(x,y);
					break;
				case "adhoc":
					this._drawAdHocMarker(x,y);
					break;
				case "compensation":
					this._drawCompensationMarker(x,y);
					break;
			}
		}
	
	});

	var GroupingElement = ElementWithBoundary.extend({
		constructor: function(structure,id,group,context,padding,spacing) {
			this.base(structure,id,group,context);
			this.padding = typeof(padding) != "undefined" ? padding : 20;
			this.spacing = typeof(spacing) != "undefined" ? spacing : 20;
			this.flowPaths = {};
			this._createElementsList(structure);
			this._rememberClickableActivities();
		},
		
		_rememberClickableActivities: function(){
			var el,m;
			
			this.clickableActivities = [];
			
			for( m in this.mapping ) {
				el = this.elements[this.mapping[m][0]][this.mapping[m][1]];
				if( el.element === "activity" && ( el.structure.href || el.structure.activities ) ) {
					this.clickableActivities.push(el);
				}
			}
		},
	
		_createElementsList: function(structure){
			this.elements = []; //vertical x horizontal
			this.mapping = {};
			var flatted = this._flattenStructure(structure);
			var flow = this._getFlow(structure,flatted);
			this.flow = $.extend({},flow);
			var i = 0;
			var f,m,e;
		
			//special elements that need to be on hand, but don't belong in the flow
			this.special = this._getSpecialElements(flatted);
			//are there some loose elements?
			var loose = this._getLooseElements(flow,flatted,this.special);
			if( !$.isEmptyObject(loose) ){
				//create invisible group for all that's inside
				if( typeof(this.elements[i]) == "undefined" ){
					this.elements[i] = [];
				}
				var substructure = this._removeLooseFromStructure(structure,flow,loose);
				var subid = 'invisible' + (new Date().getTime());
			
				if( !$.isEmptyObject(substructure.flow) ) {
					this.elements[i][0] = new GroupingElement(substructure,subid,this,this.context,0);
					this.mapping[subid] = [i,0];
					for( f in substructure.flow ) {
						delete flow[f];
						delete this.flow[f];
					}
					i++;
				}
			
				//add all loose objects
				for( m in loose ) {
					if( typeof(this.elements[i]) == "undefined" ){
						this.elements[i] = [];
					}
					this.elements[i][0] = flatted[m];
					this.mapping[m] = [i,0];
					delete flatted[m];
					this._buildTree(m,flow,flatted,1,i);
					i++;
				}
			} else if( structure.events ) {
				for( e in structure.events.start ) {
					var st = new EventElement(structure.events.start[e],e,this,this.context,"start");
					if( typeof this.elements[i] == "undefined" ){
						this.elements[i] = [];
					}
					this.elements[i][0] = st;
					this.mapping[e] = [i,0];
					i = this._buildTree(e,flow,flatted,1,i) + 1;
				}
			}
		},
	
		_popConnected: function(what,flow){
			var results = [];
			var c;
			for(c in flow){
				if( flow[c][0] == what ) {
					results.push(flow[c][1]);
					delete flow[c];
				}
			}
			return results;
		},
	
		_getFlow: function(structure,flatted) {
			var flow = $.extend({}, structure.flow);
			//add task -> boundary event connections
			var i = 0;
			var n,b,f,bi;
			for( n in structure.activities ) {
				if( structure.activities[n].boundary ) {
					for( b = 0; b < structure.activities[n].boundary.length; b++ ) {
						bi = structure.activities[n].boundary[b];
						for( f in flow ) {
							if( flow[f][0] == bi ) {
								flow[f][0] = n;
								flow[f][3] = bi;
							}
						}
						i++;
					}
				}
			}
			return flow;
		},
	
		_flattenStructure: function(structure){
			var flat = {};
			var n,b,bi;
			
			//events
			if( structure.events ) {
				for( n in structure.events.intermediate ) {
					flat[n] = new EventElement(structure.events.intermediate[n],n,this,this.context);
				}
				for( n in structure.events.end ) {
					flat[n] = new EventElement(structure.events.end[n],n,this,this.context,"end");
				}
			}
			//activities
			for( n in structure.activities ) {
				if( structure.activities[n].activities || structure.activities[n].events ) {
					flat[n] = new GroupingActivity(structure.activities[n],n,this,this.context);
				} else {
					flat[n] = new ActivityElement(structure.activities[n],n,this,this.context);
				}
				if( structure.activities[n].boundary && structure.events && structure.events.intermediate ) {
					for( b = 0; b < structure.activities[n].boundary.length; b++ ) {
						bi = structure.activities[n].boundary[b];
						flat[bi] = new EventElement(structure.events.intermediate[bi],bi,this,this.context);
						flat[bi].boundaryFor = n;
					}
				}
			}
			//gateways
			for( n in structure.gateways ) {
				flat[n] = new GatewayElement(structure.gateways[n],n,this,this.context);
			}
			return flat;
		},
	
		_buildTree: function(node,flow,flatted,hor,ver){
			var connected = this._popConnected(node,flow);
			var mv = 0;
			var i = 0;
			var n, el;
		
			for( n = 0; n < connected.length; n++ ) {
				el = flatted[connected[n]];
				if( typeof el != "undefined" ) {
					if( typeof this.elements[ver+i] == "undefined") {
						this.elements[ver+i] = [];
					} else {
						while(typeof this.elements[ver+i][hor] != "undefined" ) {
							i++;
							if( typeof this.elements[ver+i] == "undefined") {
								this.elements[ver+i] = [];
							}
						}
					}
					this.elements[ver+i][hor] = el;
					this.mapping[connected[n]] = [ver+i, hor];
					i++;
				} else if ( !this.mapping[connected[n]] ) {
					throw "Cannot find element to connect to: "+connected[n];
				}
			}

			//build tree for each child
			i = 0;
			for( n = 0; n < connected.length; n++ ) {
				if( typeof flatted[connected[n]] != "undefined" ) {
					delete flatted[connected[n]];
					mv = Math.max(this._buildTree(connected[n],flow,flatted,hor+1,ver+i),mv);
					i++;
				}
			}
		
			return Math.max(mv,ver+i);
		},
	
		_getSpecialElements: function(flatted){
			var special = $.extend({},flatted);
			var e;
			for( e in special ) {
				if( !special[e].boundaryFor ) {
					delete special[e];
				}
			}
			return special;
		},
	
		_getLooseElements: function(flow,flatted,special){
			var loose = $.extend({},flatted);
			var f = $.extend({},flow);
			var c,e;
			for ( c in f ){
				delete loose[f[c][1]];
			}
			for ( e in special ){
				delete loose[e];
			}
			return loose;
		},
	
		_removeLooseFromStructure: function(structure,flow,loose) {
			var s = $.extend({},structure);
			var f = $.extend({},flow);
			var e;
			var delLoose = function(node,s,f){
				var c;
				//events
				if( s.events ) {
					if( s.events.start ) { delete s.events.start[node]; }
					if( s.events.intermediate ) { delete s.events.intermediate[node]; }
					if( s.events.end ) { delete s.events.end[node]; }
				}
				//activities
				if( s.activities ) { delete s.activities[node]; }
				//gateways
				if( s.gateways ) { delete s.gateways[node]; }
				var con = this._popConnected(node,f);
				for ( c in s.flow ) {
					if( s.flow[c][0] == node ) {
						delete s.flow[c];
					}
				}
				for( c = 0; c < con.length; c++ ) {
					delLoose.call(this,con[c],s,f);
				}
			};
			for ( e in loose ){
				delLoose.call(this,e,s,f);
			}
			delete s.name;
			delete s.boundary;
			return s;
		},
	
		_calculatePositions: function(){
			var maxHeights = [];
			var maxWidths = [];
			var maxHeightsNoAnnots = [];
			var i,j,bb,el,yo,sumH,sumW;
			for( i = 0; i < this.elements.length; ++i) {
				maxHeights.push(0);
				maxHeightsNoAnnots.push(0);
				if( this.elements[i] ) {
					for( j = 0; j < this.elements[i].length; ++j) {
						if( typeof this.elements[i][j] != "undefined" ) {
							if( this.elements[i][j]._calculatePositions ) {
								this.elements[i][j]._calculatePositions();
							}
							bb = this.elements[i][j].getBoundingBox(true);
							maxHeights[i] = Math.max(maxHeights[i],bb[3]);
							maxHeightsNoAnnots[i] = Math.max(maxHeightsNoAnnots[i],this.elements[i][j].getBoundingBox()[3]);
							if( typeof maxWidths[j] == "undefined" ) {
								maxWidths[j] = 0;
							}
							maxWidths[j] = Math.max(maxWidths[j],bb[2]);
						}
					}
				}
			}
		
			sumH = 0;
			for( i = 0; i < this.elements.length; ++i) {
				sumW = 0;
				if( this.elements[i] ) {
					for( j = 0; j < this.elements[i].length; ++j) {
						el = this.elements[i][j];
						if( typeof el != "undefined" ) {
							bb = el.getBoundingBox();
							yo = (maxHeightsNoAnnots[i] - bb[3])/2;
							el.setOffsetX(sumW);
							el.setOffsetY(sumH+yo);
						}
						sumW += maxWidths[j]+2*this.spacing;
					}
				}
				sumH += maxHeights[i]+2*this.spacing;
			}
		},
	
		_getChildrenBoundingBox: function(){
			var x = 0;
			var y = 0;
			var w = 0;
			var h = 0;
			var k,i,j,el,bb,jTab;
		
			var iTab = [0, this.elements.length-1];
			for( k = 0; k < 2; ++k ) {
				i = iTab[k];
				if( this.elements[i] ) {
					for( j = 0; j < this.elements[i].length; ++j) {
						el = this.elements[i][j];
						if( typeof el != "undefined" ) {
							bb = el.getBoundingBox(true);
							x = Math.min(x,bb[0]);
							y = Math.min(y,bb[1]);
							w = Math.max(w,bb[0]+bb[2]);
							h = Math.max(h,bb[1]+bb[3]);
						}
					}
				}
			}
		
			for(i = 1; i < this.elements.length - 1; ++i) {
				if( this.elements[i] ) {
					jTab = [0, this.elements[i].length-1];
					for( k = 0; k < 2; k++) {
						j = jTab[k];
						el = this.elements[i][j];
						if( typeof el != "undefined" ) {
							bb = el.getBoundingBox(true);
							x = Math.min(x,bb[0]);
							y = Math.min(y,bb[1]);
							w = Math.max(w,bb[0]+bb[2]);
							h = Math.max(h,bb[1]+bb[3]);
						}
					}
				}
			}
		
			return [x,y,w,h];
		},
	
		getBoundaryEvents: function(boundingBox){
			var t,b,bi,el;
			if( !this.boundary ) {
				this.boundary = {};
				if( this.structure.boundary ) {
					t = this.getName();
					boundingBox = boundingBox || this.getBoundingBox();
					var h = boundingBox[3];
					for( b = 0; b < this.structure.boundary.length; b++ ) {
						bi = this.structure.boundary[b];
						if( this.group && this.group.special && this.group.special[bi] ) {
							el = this.group.special[bi];
							el.setOffsetX(boundingBox[2]-(b+1)*100);
							el.setOffsetY(h-65);
							this.boundary[el.id] = el;
						}
					}
				}
			}
			return this.boundary;
		},

		getElement: function(name){
			var m = this.mapping[name];
			var n,el,c;
			if( m ) {
				return this.elements[m[0]][m[1]];
			}
			m = this.special[name];
			if( m ) {
				return m;
			}

			for ( n in this.mapping ) {
				m = this.mapping[n];
				el = this.elements[m[0]][m[1]];
				if( el.getElement ){
					c = el.getElement(name);
					if ( c ) {
						return c;
					}
				}
			}
		
			return false;
		},
	
		getBoundingBox: function(withAnnots){
			var cBB = this._getChildrenBoundingBox();
			var bb = [this.offsetX,this.offsetY,cBB[2]+2*this.padding,cBB[3]+2*this.padding];
			var mh,b,nh;
			nh = this.structure.markers && this.structure.markers.length ? 10 : 0;
			if( this.structure.name ) {
				bb[3] += this.context.getTextHeight(this.structure.name,cBB[2])+this.spacing;
			}
			if( withAnnots ) {
				mh = 0;
				nh = 0;
				for( b in this.getBoundaryEvents(bb) ) {
					mh = Math.max(mh,this.boundary[b].getBoundingBox(true)[3]);
				}
				bb[3] += mh;
			} else if ( !$.isEmptyObject(this.getBoundaryEvents(bb)) ) {
				bb[3] += 25;
			}
			bb[3] += nh;
			return bb;
		},
	
		_drawName: function(){
			if( this.structure.name ) {
				var bb = this.getBoundingBox(true);
				this.context.textAlign = 'start';
				this.context.fillWrapText(this.structure.name,0,0,bb[2]-2*this.padding);
				this.context.translate(0,this.context.getTextHeight(this.structure.name,bb[2]-2*this.padding)+this.spacing);
				this.context.textAlign = 'center';
			} 
		},
	
		_drawElements: function(debug){
			var i,j,el;
			for( i = 0; i < this.elements.length; ++i) {
				if( this.elements[i] ) {
					for( j = 0; j < this.elements[i].length; ++j) {
						el = this.elements[i][j];
						if( typeof el != "undefined" ) {
							el.draw(debug);
						}
					}
				}
			}
		},
	
		_drawBoundary: function(debug){
			var b;
			for( b in this.getBoundaryEvents() ) {
				this.boundary[b].draw(debug);
			}
		},
	
		_drawDebug: function(debug){
			if( debug ) {
				this.context.save();
				this.context.textAlign = 'start';
				this.context.translate(-15,-15);
				this.context.fillStyle = 'red';
				this.context.fillText(this.id, 0, 0);
				this.context.restore();
			}
		},
	
		_getConnectionEndpoints: function(connection,debug) {
			//get bounding box of connected elements
			var ma = this.mapping[this.flow[connection][0]];
			var el = this.elements[ma[0]][ma[1]];
			var ma2 = this.mapping[this.flow[connection][1]];
			var el2 = this.elements[ma2[0]][ma2[1]];
			var bb = el.getBoundingBox();
			var bb2 = el2.getBoundingBox();
		
			var ep = el.getPossibleEndpoints(true,this.flow[connection],this.spacing/2);
			var ep2 = el2.getPossibleEndpoints(false,this.flow[connection],this.spacing/2);
		
			var xstart, ystart,
				xend, yend,
				xstartm, ystartm,
				xendm, yendm;
			var min = Number.MAX_VALUE;
			
			var i,j,d,xdiff,ydiff;
			for( i = 0; i < ep.length; i++ ) {
				for( j = 0; j < ep2.length; j++ ) {
					d = Geometry.distance(ep[i],ep2[j]);
					xdiff = Math.abs(ep2[j][0] - ep[i][0]);
					ydiff = Math.abs(ep2[j][1] - ep[i][1]);
					if( ( bb[1] + bb[3]/2 != bb2[1] + bb2[3]/2 ) && (
						( ep2[j][0] > ep[i][0] && ep2[j][1] < ep[i][1] && ep[i][1] == ep[i][3] && ep2[j][0] == ep2[j][2] ) ||
						( ep2[j][0] > ep[i][0] && ep2[j][1] > ep[i][1] && ep[i][0] == ep[i][2] && ep2[j][1] == ep2[j][3] ) ) ) {
						d -= ep[i][1] == ep[i][3] ? bb2[2]/2 : bb2[3]/2;
					}
					if( d < min ) {
						min = d;
						xstart = ep[i][0];
						ystart = ep[i][1];
						xend = ep2[j][0];
						yend = ep2[j][1];
						xstartm = ep[i][2];
						ystartm = ep[i][3];
						xendm = ep2[j][2];
						yendm = ep2[j][3];
					}
				}
			}
		
			//draw debug points
			if( debug ) {
				this.context.save();
				this.context.fillStyle = '#ff0000';
				this.context.beginPath();
				this.context.arc(xstart,ystart,4,0,2*Math.PI);
				this.context.arc(xstartm,ystartm,2,0,2*Math.PI);
				this.context.arc(xendm,yendm,2,0,2*Math.PI);
				this.context.arc(xend,yend,4,0,2*Math.PI);
				this.context.fill();
				this.context.restore();
			}
		
			return [ [xstart,ystart], [xstartm,ystartm], [xendm,yendm], [xend,yend] ];
		},
	
		/**
		* Checks if line collides with any element on canvas
		*/
		_collides: function(start,end) {
			var m,ma,el,b,c;
			for( m in this.mapping ) {
				ma = this.mapping[m];
				el = this.elements[ma[0]][ma[1]];
				b = el.getBoundingBox();
			
				c = Geometry.inside(b,start) || Geometry.inside(b,end) ||
					Geometry.cross(start,end,[b[0],b[1]], [b[0]+b[2],b[1]]) ||
					Geometry.cross(start,end,[b[0],b[1]], [b[0],b[1]+b[3]]) ||
					Geometry.cross(start,end,[b[0]+b[2],b[1]+b[3]], [b[0]+b[2],b[1]]) ||
					Geometry.cross(start,end,[b[0]+b[2],b[1]+b[3]], [b[0],b[1]+b[3]]);

				if( c ) {
					return el.getBoundingBox(true);   
				}
			}
			return false;
		},
	
		//prefOrientation == false - horizontal
		_findPath: function(start,end,prefOrientation,rs,debug,i){
			var path = [];
			i = typeof(i) != "undefined" ? i : 50;
		
			if( Geometry.distance(start,end) > 1 && i ) {
			
				if( ( !prefOrientation && end[0] != start[0] ) || end[1] == start[1] ) {
					mx = start[0];
					my = end[1];
				} else {
					mx = end[0];
					my = start[1];
				}
			
				var b = this._collides([mx,my],end);
				if( b ) {
					if( !prefOrientation || Math.abs(end[1] - my) < 1 ) {
						if( start[0] <= end[0] ) {
							mx = b[0] + b[2] + rs;
						} else {
							mx = b[0] - rs;
						}
					}
					if ( prefOrientation || Math.abs(end[0] - mx) < 1 ) {
						if( start[1] < end[1] ) {
							my = b[1] + b[3] + rs;
						} else {
							my = b[1] - rs;
						}
						mx -= mx - end[0];
					}
					if( !prefOrientation || Math.abs(end[1] - my) < 1 ) {
						my -= my - end[1];
					}
				}
		
				path = this._findPath(start,[mx,my],!prefOrientation,rs,debug,i-1);
			}
		
			path.push(end);
			return path;
		},
	
		_drawConnections: function(debug){
			var f,endpoints,path,i,el,p,h,tx,ty,ma;
			
			this.context.save();
		
			for ( f in this.flow ){
				endpoints = this._getConnectionEndpoints(f,debug);
				path = this._findPath(
					endpoints[1],endpoints[2],
					endpoints[3][1] != endpoints[2][1],
					this.spacing/2 + Math.random()*this.spacing/2,
					debug);
			
				//draw connection line
				this.context.beginPath();
				this.context.moveTo(endpoints[0][0],endpoints[0][1]);
				this.context.lineTo(endpoints[1][0],endpoints[1][1]);
				for( i = 0; i < path.length; i++ ) {
					this.context.lineTo(path[i][0],path[i][1]);
				}
				this.context.lineTo(endpoints[3][0],endpoints[3][1]);
			
				this.context.lineWidth = 5;
				this.context.strokeStyle = "#fff";
				this.context.stroke();
			
				this.context.lineWidth = 1;
				this.context.strokeStyle = "#000";
				if( this.flow[f][3] ) {
					el = this.special[ this.flow[f][3] ];
					if( el.structure.type == "compensation"  ) {
						if( this.context.setLineDash ) {
							this.context.setLineDash([1,3]);
						} else {
							this.context.strokeStyle = '#ccc';
						}
					}
				}
				this.context.stroke();
			
				//draw arrow head
				this.context.fillStyle = "#000";
				this.context.beginPath();
				this.context.moveTo(endpoints[3][0],endpoints[3][1]);
				if( endpoints[3][1] == endpoints[2][1] ) {
					p = endpoints[3][0]+this.spacing*0.4*( endpoints[3][0] > endpoints[2][0] ? -1 : 1 );
					this.context.lineTo(p,endpoints[3][1]+this.spacing/4);
					this.context.lineTo(p,endpoints[3][1]-this.spacing/4);
				} else {
					p = endpoints[3][1]+this.spacing*0.4*( endpoints[3][1] > endpoints[2][1] ? -1 : 1 );
					this.context.lineTo(endpoints[3][0]+this.spacing/4,p);
					this.context.lineTo(endpoints[3][0]-this.spacing/4,p);
				}
				this.context.lineTo(endpoints[3][0],endpoints[3][1]);
				this.context.fill();

				if( this.flow[f][2] && this.flow[f][2] !== "true" ) {
					//draw condition text
					h = this.context.getTextHeight(this.flow[f][2],100);
					tx = (path[1][0] + endpoints[1][0])/2;
					ty = (path[1][1] + endpoints[1][1])/2 - h;
				
				
					this.context.fillStyle = "rgba(255,255,255,0.5)";
					this.context.fillRect(tx-50,ty,100,h);
					this.context.textAlign = 'center';
					this.context.fillStyle = "#000";
					this.context.fillWrapText(this.flow[f][2],tx,ty,100);
				
					//if not gateway then draw diamond
					ma = this.mapping[this.flow[f][0]];
					el = this.elements[ma[0]][ma[1]];
					if( el.element != "gateway" ) {
						this.context.beginPath();
						this.context.moveTo(endpoints[0][0],endpoints[0][1]);
						if(endpoints[0][1] == endpoints[1][1]) {
							//horizontal
							this.context.lineTo((endpoints[0][0]+endpoints[1][0])/2,endpoints[0][1]-5);
							this.context.lineTo(endpoints[1][0],endpoints[1][1]);
							this.context.lineTo((endpoints[0][0]+endpoints[1][0])/2,endpoints[0][1]+5);
						} else {
							//vertical
							this.context.lineTo(endpoints[0][0]-5,(endpoints[0][1]+endpoints[1][1])/2);
							this.context.lineTo(endpoints[1][0],endpoints[1][1]);
							this.context.lineTo(endpoints[0][0]+5,(endpoints[0][1]+endpoints[1][1])/2);
						}
						this.context.closePath();
						this.context.fillStyle = "#fff";
						this.context.fill();
						this.context.stroke();
					}
				} else if ( this.flow[f][2] ) {
					//draw default marker
					this.context.beginPath();
					this.context.moveTo(path[0][0]-5,path[0][1]-5);
					this.context.lineTo(path[0][0]+5,path[0][1]+5);
					this.context.lineWidth = 2;
					this.context.stroke();
				}
			}
		
			this.context.restore();
		},
	
		draw: function(debug){
			var diagram = this;
			this.context.save();
		
			this.context.translate(this.offsetX+this.padding,this.offsetY+this.padding);
		
			this._drawName();
			this._drawConnections(debug);
			this._drawElements(debug);
		
			this._drawBoundary(debug);
		
			this._drawDebug(debug);
			this.context.restore();
		},
		
		/**
		 * Checks if the point is in the clickable activity (the one possesing
		 * href atribute in its structure) and if so returns it.
		 *
		 * @param Array point [x,y] point to check
		 * @returns The most nested clickable activity the point is in or false
		 */
		overClickableActivity: function(point) {
			var i,bb,found;
			if( this.padding ) {
				point[0] -= this.padding;
				point[1] -= this.padding;
			}
			if( this.structure.name ) {
				var tbb = this.getBoundingBox();
				point[1] -= this.context.getTextHeight(this.structure.name,tbb[2]-2*(this.padding||0))+this.spacing;
			}
			for( i = 0; i < this.clickableActivities.length; i++ ) {
				if( this.clickableActivities[i].overClickableActivity ) {
					found = this.clickableActivities[i].overClickableActivity([
						point[0] - this.clickableActivities[i].offsetX,
						point[1] - this.clickableActivities[i].offsetY
					]);
					if( found ) {
						return found;
					}
				}
				if( this.clickableActivities[i].structure.href ) {
					bb = this.clickableActivities[i].getBoundingBox(true);
					if( Geometry.inside(bb,point) ) {
						return this.clickableActivities[i];
					}
				}
			}
			return false;
		}
	});

	/* Element representing event */
	var EventElement = Element.extend({
		constructor: function(structure,id,group,context,position) {
			this.base(structure,id,group,context);
			this.position = (typeof position != "undefined") ? position : "intermediate";
			this.boundaryFor = null;
			this.element = "event";
		},
	
		getBoundingBox: function(withAnnots){
			var w = 50;
			var h = 50;
			var ao = w/2;
			var t;
			if( withAnnots ) {
				t = this.getName();
				w = 100;
				h += this.context.getTextHeight(t,w);
				ao = 0;
			}
			return [this.offsetX+ao,this.offsetY,w,h];
		},
	
		_drawMessage: function(invert){
			this.context.save();
		
			if(invert) {
				this.context.fillStyle = '#000';
				this.context.strokeStyle = '#fff';
			} else {
				this.context.fillStyle = '#fff';
				this.context.strokeStyle = '#000';
			}
		
			this.context.beginPath();
			this.context.moveTo(-12,-9);
			this.context.lineTo(12,-9);
			this.context.lineTo(12,9);
			this.context.lineTo(-12,9);
			this.context.closePath();
			this.context.fill();
			this.context.stroke();
		
			this.context.beginPath();
			this.context.moveTo(-12,-9);
			this.context.lineTo(0,3);
			this.context.lineTo(12,-9);
			this.context.stroke();
			this.context.restore();
		},
	
		_drawTimer: function(){
			this.context.save();
			this.context.fillStyle = '#fff';
			this.context.strokeStyle = '#000';
			this.context.beginPath();
			this.context.arc(0,0,18,0,2*Math.PI);
		
			this.context.moveTo(0,-14); //12
			this.context.lineTo(0,-18);
			this.context.moveTo(7,-12); //1
			this.context.lineTo(8,-15);
			this.context.moveTo(12,-7); //2
			this.context.lineTo(15,-8);
			this.context.moveTo(14,0); //3
			this.context.lineTo(18,0);
			this.context.moveTo(12,7); //4
			this.context.lineTo(15,8);
			this.context.moveTo(7,12); //5
			this.context.lineTo(8,15);
			this.context.moveTo(0,14); //6
			this.context.lineTo(0,18);
			this.context.moveTo(-7,12); //7
			this.context.lineTo(-8,15);
			this.context.moveTo(-12,7); //8
			this.context.lineTo(-15,8);
			this.context.moveTo(-14,0); //9
			this.context.lineTo(-18,0);
			this.context.moveTo(-12,-7); //10
			this.context.lineTo(-15,-8);
			this.context.moveTo(-7,-12); //11
			this.context.lineTo(-8,-15);
		
			this.context.moveTo(0,0);
			this.context.lineTo(12,0);
			this.context.moveTo(0,0);
			this.context.lineTo(3,-16);
		
			this.context.stroke();
			this.context.restore();
		},
	
		_drawError: function(invert){
			this.context.save();
		
			if(invert) {
				this.context.fillStyle = '#000';
				this.context.strokeStyle = '#fff';
			} else {
				this.context.fillStyle = '#fff';
				this.context.strokeStyle = '#000';
			}
		
			this.context.beginPath();
			this.context.moveTo(-14,14);
			this.context.lineTo(-7,-11);
			this.context.lineTo(4,0);
			this.context.lineTo(14,-14);
			this.context.lineTo(7,11);
			this.context.lineTo(-4,0);
			this.context.closePath();
			this.context.fill();
			this.context.stroke();
			this.context.restore();
		},
	
		_drawCompensation: function(invert){
			this.context.save();
		
			if(invert) {
				this.context.fillStyle = '#000';
				this.context.strokeStyle = '#fff';
			} else {
				this.context.fillStyle = '#fff';
				this.context.strokeStyle = '#000';
			}
		
			this.context.beginPath();
		
			this.context.moveTo(-14,0);
			this.context.lineTo(-2,-9);
			this.context.lineTo(-2,9);
			this.context.lineTo(-14,0);
			this.context.moveTo(-2,0);
			this.context.lineTo(10,-9);
			this.context.lineTo(10,9);
			this.context.lineTo(-2,0);
		
			this.context.closePath();
			this.context.fill();
			this.context.stroke();
			this.context.restore();
		},
	
		_drawEscalation: function(invert){
			this.context.save();
		
			if(invert) {
				this.context.fillStyle = '#000';
				this.context.strokeStyle = '#fff';
			} else {
				this.context.fillStyle = '#fff';
				this.context.strokeStyle = '#000';
			}
		
			this.context.beginPath();
		
			this.context.moveTo(-14,10);
			this.context.lineTo(0,-16);
			this.context.lineTo(14,10);
			this.context.lineTo(0,0);
		
			this.context.closePath();
			this.context.fill();
			this.context.stroke();
			this.context.restore();
		},
	
		_drawConditional: function(){
			this.context.save();
		
			this.context.fillStyle = '#fff';
			this.context.strokeStyle = '#000';
			this.context.beginPath();
		
			this.context.moveTo(-12,14);
			this.context.lineTo(-12,-14);
			this.context.lineTo(12,-14);
			this.context.lineTo(12,14);
		
			this.context.closePath();
			this.context.fill();
			this.context.stroke();
	
			this.context.moveTo(-10,-9.5);
			this.context.lineTo(10,-9.5);
			this.context.moveTo(-10,-4.5);
			this.context.lineTo(10,-4.5);
			this.context.moveTo(-10,1.5);
			this.context.lineTo(10,1.5);
			this.context.moveTo(-10,7.5);
			this.context.lineTo(10,7.5);
		
			this.context.stroke();
		
			this.context.restore();
		},
	
		_drawLink: function(invert){
			this.context.save();
		
			if(invert) {
				this.context.fillStyle = '#000';
				this.context.strokeStyle = '#fff';
			} else {
				this.context.fillStyle = '#fff';
				this.context.strokeStyle = '#000';
			}
		
			this.context.beginPath();
		
			this.context.moveTo(-10,-5);
			this.context.lineTo(3,-5);
			this.context.lineTo(3,-12);
			this.context.lineTo(12,0);
			this.context.lineTo(3,12);
			this.context.lineTo(3,5);
			this.context.lineTo(-10,5);
		
			this.context.closePath();
			this.context.fill();
			this.context.stroke();
			this.context.restore();
		},
	
		_drawCancel: function(invert){
			this.context.save();
		
			if(invert) {
				this.context.fillStyle = '#000';
				this.context.strokeStyle = '#fff';
			} else {
				this.context.fillStyle = '#fff';
				this.context.strokeStyle = '#000';
			}
		
			this.context.rotate(Math.PI/4);
		
			this.context.beginPath();
		
			this.context.moveTo(-2.5,-14.5);
			this.context.lineTo(2.5,-14.5);
			this.context.lineTo(2.5,-2.5);
			this.context.lineTo(14.5,-2.5);
			this.context.lineTo(14.5,2.5);
			this.context.lineTo(2.5,2.5);
			this.context.lineTo(2.5,14.5);
			this.context.lineTo(-2.5,14.5);
			this.context.lineTo(-2.5,2.5);
			this.context.lineTo(-14.5,2.5);
			this.context.lineTo(-14.5,-2.5);
			this.context.lineTo(-2.5,-2.5);
		
			this.context.closePath();
			this.context.fill();
			this.context.stroke();
			this.context.restore();
		},
	
		_drawSignal: function(invert){
			this.context.save();
		
			if(invert) {
				this.context.fillStyle = '#000';
				this.context.strokeStyle = '#fff';
			} else {
				this.context.fillStyle = '#fff';
				this.context.strokeStyle = '#000';
			}
		
			this.context.beginPath();
		
			this.context.moveTo(-14,10.5);
			this.context.lineTo(0,-16);
			this.context.lineTo(14,10.5);
		
			this.context.closePath();
			this.context.fill();
			this.context.stroke();
			this.context.restore();
		},
	
		_drawMultiple: function(invert){
			this.context.save();
		
			if(invert) {
				this.context.fillStyle = '#000';
				this.context.strokeStyle = '#fff';
			} else {
				this.context.fillStyle = '#fff';
				this.context.strokeStyle = '#000';
			}
		
			this.context.beginPath();
		
			this.context.moveTo(0,-15);
			this.context.lineTo(14,-3);
			this.context.lineTo(9,11.5);
			this.context.lineTo(-9,11.5);
			this.context.lineTo(-14,-3);
		
			this.context.closePath();
			this.context.fill();
			this.context.stroke();
			this.context.restore();
		},
	
		_drawParallel: function(){
			this.context.save();
		
			this.context.fillStyle = '#fff';
			this.context.strokeStyle = '#000';
		
			this.context.beginPath();
		
			this.context.moveTo(-3.5,-14.5);
			this.context.lineTo(3.5,-14.5);
			this.context.lineTo(3.5,-3.5);
			this.context.lineTo(14.5,-3.5);
			this.context.lineTo(14.5,3.5);
			this.context.lineTo(3.5,3.5);
			this.context.lineTo(3.5,14.5);
			this.context.lineTo(-3.5,14.5);
			this.context.lineTo(-3.5,3.5);
			this.context.lineTo(-14.5,3.5);
			this.context.lineTo(-14.5,-3.5);
			this.context.lineTo(-3.5,-3.5);
		
			this.context.closePath();
			this.context.fill();
			this.context.stroke();
			this.context.restore();
		},
	
		_drawTerminate: function(){
			this.context.save();
		
			this.context.fillStyle = '#000';
		
			this.context.beginPath();
		
			this.context.arc(0,0,17,0,Math.PI*2);
		
			this.context.fill();
			this.context.stroke();
			this.context.restore();
		},
	
		draw: function(debug){
			this.context.save();
			this.context.translate(this.offsetX+50,this.offsetY+25);
		
			this.context.save();
			this.context.beginPath();
			this.context.arc(0,0,25,0,2*Math.PI);
			this.context.fillStyle = this.position == "end" ? '#000' : '#fff';
			this.context.fill();
		
			if( this.position != "end" &&
				(this.structure.nonInterrupting || this.structure.noninterrupting ) &&
				$.inArray(this.structure.type,[
					"message",
					"timer",
					"escalation",
					"conditional",
					"signal",
					"multiple",
					"parallel"
				]) > -1 &&
				( this.boundaryFor || this.group.structure.activityType == "event" ) ) {
				if( this.context.setLineDash ) {
					this.context.setLineDash([10,5]);
				} else {
					this.context.strokeStyle = '#ccc';
				}
			}
			if (this.position != "start") {
				this.context.stroke();
				this.context.beginPath();
				this.context.arc(0,0,22,0,2*Math.PI);
				this.context.fillStyle = '#fff';
				this.context.fill();
			}
			this.context.stroke();
			this.context.restore();
		
			switch( this.structure.type ) {
				case "message":
					this._drawMessage(this.position == "end" || this.structure.throwing);
					break;
				case "timer":
					this._drawTimer();
					break;
				case "error":
					this._drawError(this.position == "end");
					break;
				case "compensation":
					this._drawCompensation(this.position == "end" || this.structure.throwing);
					break;
				case "escalation":
					this._drawEscalation(this.position == "end" || this.structure.throwing);
					break;
				case "conditional":
					this._drawConditional();
					break;
				case "link":
					this._drawLink(this.structure.throwing);
					break;
				case "cancel":
					this._drawCancel(this.position == "end");
					break;
				case "signal":
					this._drawSignal(this.position == "end" || this.structure.throwing);
					break;
				case "multiple":
					this._drawMultiple(this.position == "end" || this.structure.throwing);
					break;
				case "parallel":
					this._drawParallel();
					break;
				case "terminate":
					this._drawTerminate();
					break;
			}
		
			var t = this.getName();
			this.context.fillStyle = 'rgba(255,255,255,0.5)';
			this.context.fillRect(-50,28,100,this.context.getTextHeight(t,100));
			this.context.fillStyle = '#000';
			this.context.fillWrapText(t,0,30,100);
		
			if( debug ) {
				this.context.fillStyle = 'red';
				this.context.fillText(this.id, 0, 0);
			}
		
			this.context.restore();
		}
	});

	/* Element representing task */
	var ActivityElement = ElementWithBoundary.extend({

		constructor: function(structure,id,group,context) {
			this.base(structure,id,group,context);
			this.element = "activity";
		},

		getBoundaryEvents: function(){
			var t,h,b,bi,el;
		
			if( !this.boundary ) {
				this.boundary = {};
				if( this.structure.boundary ) {
					t = this.getName();
					h = this.context.getTextHeight(t,100) + 10;
					for( b = 0; b < this.structure.boundary.length; b++ ) {
						bi = this.structure.boundary[b];
						if( this.group && this.group.special && this.group.special[bi] ) {
							el = this.group.special[bi];
							el.setOffsetX(Math.max(140,100*this.structure.boundary.length-40)-(b+1)*100);
							el.setOffsetY(h);
							this.boundary[el.id] = el;
						}
					}
				}
			}
			return this.boundary;
		},
	
		getBoundingBox: function(withAnnots){
			var t = this.getName();
			var h = this.context.getTextHeight(t,100);
			var w = 140;
			var mh,b,nh;
		
			nh = this.structure.markers && this.structure.markers.length ? 10 : 0;
		
			if ( !$.isEmptyObject(this.getBoundaryEvents()) ) {
				mh = 0;
				nh = 0;
				for( b in this.getBoundaryEvents() ) {
					mh = Math.max(mh,this.boundary[b].getBoundingBox(withAnnots)[3]);
				}
				if( withAnnots ) {
					h += mh;
				} else {
					h += 25;
				}
				w = Math.max(w,100*this.structure.boundary.length-40);
			}
			return [this.offsetX,this.offsetY,w,h+nh+40];
		},
		
		_drawMessage: function(invert,instantiate){
			this.context.save();
		
			if( invert ) {
				this.context.fillStyle = '#000';
				this.context.strokeStyle = '#fff';
			} else {
				this.context.fillStyle = '#fff';
				this.context.strokeStyle = '#000';
			}
			
			if( !instantiate ) {
				this.context.beginPath();
				this.context.moveTo(-14,-14);
				this.context.lineTo(2,-14);
				this.context.lineTo(2,-2);
				this.context.lineTo(-14,-2);
				this.context.closePath();
				this.context.fill();
				this.context.stroke();
			
				this.context.beginPath();
				this.context.moveTo(-14,-14);
				this.context.lineTo(-6,-6);
				this.context.lineTo(2,-14);
				this.context.stroke();
			} else {
				this.context.beginPath();
				this.context.moveTo(4,-7.5);
				this.context.arc(-6,-7.5,10,0,2*Math.PI);
				this.context.stroke();
			
				this.context.beginPath();
				this.context.moveTo(-12,-12.5);
				this.context.lineTo(0,-12.5);
				this.context.lineTo(0,-2.5);
				this.context.lineTo(-12,-2.5);
				this.context.closePath();
				this.context.fill();
				this.context.stroke();
			
				this.context.beginPath();
				this.context.moveTo(-12,-12.5);
				this.context.lineTo(-6,-6);
				this.context.lineTo(0,-12.5);
				this.context.stroke();
			}
		
			this.context.restore();
		},
	
		_drawService: function(){
		
			var drawGearWheel = function(x,y,r1,r2){
				var i,m;
				
				this.context.save();
				this.context.beginPath();
				this.context.moveTo(r2+x,y);
			
				for(i = 0; i < 8; i++) {
					m = i*Math.PI/4;
					this.context.arc(x,y,r2,m,m+Math.PI/30);
					this.context.arc(x,y,r1,m+Math.PI/12,m+Math.PI/6);
					this.context.arc(x,y,r2,m+39*Math.PI/180,m+Math.PI/4);
				}
			
				this.context.stroke();
				this.context.fill();
				this.context.restore();
			};
		
			this.context.save();
		
			this.context.fillStyle = '#fff';

			drawGearWheel.call(this,-6,-6,5,8);
			drawGearWheel.call(this,-6,-6,1,2);
		
			drawGearWheel.call(this,-1,-1,5,8);
			drawGearWheel.call(this,-1,-1,1,2);
		
			this.context.restore();
		},
	
		_drawUser: function(){
			this.context.save();
	
			this.context.beginPath();
			this.context.moveTo(0,-1);
			this.context.lineTo(0,2);
			this.context.lineTo(-12,2);
			this.context.lineTo(-12,-1);
			this.context.arc(-6,-1,6,Math.PI,2*Math.PI);
			this.context.stroke();
	
			this.context.beginPath();
			this.context.arc(-6,-10,3,0,2*Math.PI);
			this.context.stroke();
		
			this.context.restore();
		},
	
		_drawManual: function(){
			this.context.save();
			this.context.translate(-14,-12);
	
			this.context.beginPath();
			this.context.moveTo(0,3);
			this.context.lineTo(3,1);
			this.context.lineTo(9,0);
			this.context.lineTo(10,2);
			this.context.lineTo(6,3);
			this.context.lineTo(15,3);
			this.context.lineTo(15,5);
			this.context.lineTo(16,5);
			this.context.lineTo(16,7);
			this.context.lineTo(15,7);
			this.context.lineTo(15,9);
			this.context.lineTo(13,9);
			this.context.lineTo(13,11);
			this.context.lineTo(3,11);
			this.context.lineTo(0,10);
			this.context.closePath();
			this.context.stroke();
	
			this.context.beginPath();
			this.context.moveTo(15,5);
			this.context.lineTo(8,5);
			this.context.moveTo(15,7);
			this.context.lineTo(8,7);
			this.context.moveTo(13,9);
			this.context.lineTo(7,9);
	
			this.context.lineWidth = 0.5;
			this.context.stroke();
	
			this.context.restore();
		},
	
		_drawBusiness: function(){
			this.context.save();
	
			this.context.beginPath();
			this.context.moveTo(-14,-14);
			this.context.lineTo(2,-14);
			this.context.lineTo(2,-2);
			this.context.lineTo(-14,-2);
			this.context.closePath();
			this.context.stroke();
	
			this.context.beginPath();
			this.context.moveTo(-14,-10);
			this.context.lineTo(2,-10);
			this.context.moveTo(-14,-6);
			this.context.lineTo(2,-6);
			this.context.moveTo(-9,-10);
			this.context.lineTo(-9,-2);
			this.context.stroke();
	
			this.context.restore();
		},
	
		_drawScript: function(){
			this.context.save();
		
			this.context.beginPath();
			this.context.moveTo(2,-14);
			this.context.lineTo(-10,-14);
			this.context.bezierCurveTo(-18,-11,-6,-4,-14,1);
			this.context.lineTo(-2,1);
			this.context.bezierCurveTo(6,-4,-6,-11,2,-14);
			this.context.stroke();
	
			this.context.lineWidth = 0.5;
	
			this.context.beginPath();
			this.context.moveTo(-10,-11);
			this.context.lineTo(-3,-11);
			this.context.moveTo(-9.5,-8);
			this.context.lineTo(-2.5,-8);
			this.context.moveTo(-8.5,-5);
			this.context.lineTo(-1.5,-5);
			this.context.moveTo(-9,-2);
			this.context.lineTo(-2,-2);
			this.context.stroke();
	
			this.context.restore();
		},
	
		draw: function(debug){
			var ml,m,b,t;
		
			this.context.save();
		
			var bb = this.getBoundingBox();
		
			if ( this.structure.markers ) {
				ml = this.structure.markers.length;
				for( m = 0; m < ml; m++) {
					this._drawMarker( bb[0] + bb[2]/2 - ml*10 + m*20, bb[1]+bb[3]-20, this.structure.markers[m] );
				}
			}
			
			if( this.structure.href ) {
				this.context.strokeStyle = '#060';
			}
		
			switch( this.structure.activityType ) {
				case "call":
					this.context.lineWidth = 4;
					break;
				case "transaction":
					this.context.strokeRoundedRect(bb[0]+3,bb[1]+3,bb[2]-6,bb[3]-6,12);
					break;
			}
		
			this.context.strokeRoundedRect(bb[0],bb[1],bb[2],bb[3],15);
		
			this.context.restore();
			this.context.save();
		
			this.context.translate(this.offsetX+20,this.offsetY+20);
		
			t = this.getName();
			this.context.fillWrapText(t,50+( bb[2]-140 )/2,0,100);
		
			for( b in this.getBoundaryEvents() ) {
				this.boundary[b].draw(debug);
			}
		
			if( !this.structure.markers || (this.structure.markers && $.inArray("subprocess",this.structure.markers) == -1 ) ) {
				switch( this.structure.type ) {
					case "service":
						this._drawService();
						break;
					case "send":
						this._drawMessage(true);
						break;
					case "receive":
						this._drawMessage(false,this.structure.instantiate);
						break;
					case "user":
						this._drawUser();
						break;
					case "manual":
						this._drawManual();
						break;
					case "business":
						this._drawBusiness();
						break;
					case "script":
						this._drawScript();
						break;
				}
			}
		
			if( debug ) {
				this.context.save();
				this.context.textAlign = 'start';
				this.context.translate(-15,-15);
				this.context.fillStyle = 'red';
				this.context.fillText(this.id, 0, 0);
				this.context.restore();
			}
		
			this.context.restore();
		}
	});

	/* Element representing various activities grouping other elements */
	var GroupingActivity = GroupingElement.extend({
		constructor: function(structure,id,group,context) {
			this.base(structure,id,group,context);
			this.element = "activity";
		},

		draw: function(debug){
			var ml,m;
		
			//draw border
			this.context.save();
			var bb = this.getBoundingBox();
		
			if ( this.structure.markers ) {
			
				ml = this.structure.markers.length;
				for( m = 0; m < ml; m++) {
					this._drawMarker( bb[0] + bb[2]/2 - ml*10 + m*20, bb[1]+bb[3]-20, this.structure.markers[m] );
				}
			}
		
			if( this.structure.href ) {
				this.context.strokeStyle = '#060';
			}
		
			switch( this.structure.activityType ) {
				case "event":
					if( this.context.setLineDash ) {
						this.context.setLineDash([10,5]);
					} else if ( this.structure.href ){
						this.context.strokeStyle = '#5b5';
					} else {
						this.context.strokeStyle = '#ccc';
					}
					break;
				case "call":
					this.context.lineWidth = 4;
					break;
				case "transaction":
					this.context.strokeRoundedRect(bb[0]+3,bb[1]+3,bb[2]-6,bb[3]-6,12);
					break;
			}
			this.context.strokeRoundedRect(bb[0],bb[1],bb[2],bb[3],15);
			this.context.restore();
		
			//draw content
			this.base(debug);
		}
	});

	/* Element representing gateway */
	var GatewayElement = Element.extend({
		constructor: function(structure,id,group,context) {
			this.base(structure,id,group,context);
			this.element = "gateway";
		},

		getBoundingBox: function(withAnnots){
			var w = 50;
			var h = 50;
			var ao = w/2;
			if( withAnnots ) {
				w = 100;
				ao = 0;
			}
			return [this.offsetX+ao,this.offsetY,w,h];
		},
	
		_drawInclusive: function(){
			this.context.save();
			this.context.lineWidth = 3;
			this.context.beginPath();
			this.context.arc(0,0,13,0,2*Math.PI);
			this.context.stroke();
			this.context.restore();
		},
	
		_drawComplex: function(){
			this.context.save();
			this.context.beginPath();
			this.context.moveTo(0,-15);
			this.context.lineTo(0,15);
			this.context.moveTo(-15,0);
			this.context.lineTo(15,0);
		
			this.context.moveTo(-10,-10);
			this.context.lineTo(10,10);
			this.context.moveTo(-10,10);
			this.context.lineTo(10,-10);
		
			this.context.lineWidth = 4;
			this.context.stroke();
			this.context.restore();
		},
	
		_drawParallel: function(instantiate){
			this.context.save();
			this.context.beginPath();
			if( !instantiate ) {
				this.context.moveTo(0,-15);
				this.context.lineTo(0,15);
				this.context.moveTo(-15,0);
				this.context.lineTo(15,0);
				this.context.lineWidth = 4;
			} else {
				this.context.arc(0,0,15,0,2*Math.PI);
				this.context.moveTo(-2,-12);
				this.context.lineTo(2,-12);
				this.context.lineTo(2,-2);
				this.context.lineTo(12,-2);
				this.context.lineTo(12,2);
				this.context.lineTo(2,2);
				this.context.lineTo(2,12);
				this.context.lineTo(-2,12);
				this.context.lineTo(-2,2);
				this.context.lineTo(-12,2);
				this.context.lineTo(-12,-2);
				this.context.lineTo(-2,-2);
				this.context.closePath();
			}
			this.context.stroke();
			this.context.restore();
		},
	
		_drawEvent: function(instantiate){
			this.context.save();
			this.context.beginPath();
			this.context.arc(0,0,15,0,2*Math.PI);
			if( !instantiate ) {
				this.context.stroke();
				this.context.beginPath();
				this.context.arc(0,0,12,0,2*Math.PI);
			}
			this.context.stroke();
		
			this.context.beginPath();
			this.context.moveTo(0,-9);
			this.context.lineTo(8,-1);
			this.context.lineTo(6,7);
			this.context.lineTo(-6,7);
			this.context.lineTo(-8,-1);
			this.context.closePath();
			this.context.stroke();
			this.context.restore();
		},
	
		draw: function(debug){
			this.context.save();
			this.context.translate(this.offsetX+50,this.offsetY+25);
		
			this.context.beginPath();
			this.context.moveTo(-25,0);
			this.context.lineTo(0,-25);
			this.context.lineTo(25,0);
			this.context.lineTo(0,25);
			this.context.closePath();
			this.context.stroke();
		
			switch( this.structure.type ) {
				case "inclusive":
					this._drawInclusive();
					break;
				case "complex":
					this._drawComplex();
					break;
				case "parallel":
					this._drawParallel(this.structure.instantiate);
					break;
				case "event":
					this._drawEvent(this.structure.instantiate);
					break;
			}
		
			var t = this.getName();
			var h = this.context.getTextHeight(t,100);
			this.context.fillStyle = "rgba(255,255,255,0.5)";
			this.context.fillRect(-50,-h-25,100,h);
			this.context.fillStyle = "#000";
			this.context.fillWrapText(t,0,-h-25,100);
		
			if( debug ) {
				this.context.fillStyle = 'red';
				this.context.fillText(this.id, 0, 0);
			}
		
			this.context.restore();
		}
	});

})(jQuery);
