jQuery.expr[':'].node = function(elem, index, match){
	var nn = elem.nodeName.toLowerCase().replace(/.*?:/,'');
	return nn == match[3].toLowerCase();
};

var Converter = {
	/**
	 * Initializes the converter, remembers necessary jQuery elements,
	 * binds click events
	 *
	 * @name Converter.init
	 * @type Converter
	 */
	init: function(edid){
		var self = this;
	
		this.xmlTxtarea = $('#code_xml');
		this.resTxtarea = $('#code_simple');
		this.parentTxtarea = $(window.opener.document).find('#'+edid);
		this.convertBtn = $('#convert');
		this.insertBtn = $('#insert');
		
		this.convertBtn.click(function(ev){
			ev.preventDefault();
			
			var hs = self.xmlTxtarea.height(),
				hr = self.resTxtarea.height(),
				res = self.convert(self.xmlTxtarea.val());
			
			self.resTxtarea.val(res);
			
			self.resTxtarea.height(hs);
			self.xmlTxtarea.height(hr);
			
			self.insertBtn.removeProp('disabled');
		});
		
		this.insertBtn.click(function(ev){
			ev.preventDefault();
			
			var c = self.parentTxtarea.caret();
				v = c.replace(self.resTxtarea.val());
			self.parentTxtarea.val(v);
			
			var close = confirm("Code inserted! Do you want to close this window?");
			if( close ) {
				window.close();
				self.parentTxtarea.focus().caret(c.start,c.start);
			}
		});
		
		return this;
	},
	
	/**
	 * Actual method taking care of conversion.
	 *
	 * @param String xmlString BPMN XML representation
	 *
	 * @name Converter.convert
	 * @type String
	 */
	convert: function(xmlString){
		var $xml = $($.parseXML(xmlString)),
			$process = $xml.find('*:node(process)'),
			res = this.translateFragment($process);
		
		return this.processResults(res);
	},
	
	_translateID: function(id){
		return id.replace(/^_/,'').replace(/[^a-z0-9_]/ig,'_');
	},
	
	_translateEvent: function($n,tid,res){
		var obj;
		
		/* Create ID and object in correct place */
		if( $n.is(':node(startEvent)') ) {
			tid = 'se_'+tid;
			obj = res.events.start[tid] = {};
		} else if( $n.is(':node(endEvent)') ) {
			tid = 'ee_'+tid;
			obj = res.events.end[tid] = {};
		} else {
			tid = 'ie_'+tid;
			obj = res.events.intermediate[tid] = {};
			
			/* Is boundary event? */
			if( $n.attr('attachedToRef') ) {
				obj.boundaryFor = $n.attr('attachedToRef');
			}
			
			/* Is throwing? */
			if( $n.is(':node(intermediateThrowEvent)') ) {
				obj.throwing = true;
			}
		}
		
		/* Is interrupting? */
		if( $n.attr('isInterrupting') === "false" || $n.attr('cancelActivity') === "false" ) {
			obj.nonInterrupting = true;
		}

		/* Determine type */
		if( $n.children(':node(errorEventDefinition)').length ) {
			obj.type = "error";
		} else if( $n.children(':node(compensateEventDefinition)').length ) {
			obj.type = "compensation";
		} else if( $n.children(':node(messageEventDefinition)').length ) {
			obj.type = "message";
		} else if( $n.children(':node(timerEventDefinition)').length ) {
			obj.type = "timer";
		} else if( $n.children(':node(escalationEventDefinition)').length ) {
			obj.type = "escalation";
		} else if( $n.children(':node(conditionalEventDefinition)').length ) {
			obj.type = "conditional";
		} else if( $n.children(':node(linkEventDefinition)').length ) {
			obj.type = "link";
		} else if( $n.children(':node(cancelEventDefinition)').length ) {
			obj.type = "cancel";
		} else if( $n.children(':node(signalEventDefinition)').length ) {
			obj.type = "signal";
		} else if( $n.children(':node(multipleEventDefinition)').length ) {
			obj.type = "multiple";
		} else if( $n.children(':node(parallelEventDefinition)').length ) {
			obj.type = "parallel";
		} else if( $n.children(':node(terminateEventDefinition)').length ) {
			obj.type = "terminate";
		}
		
		return [tid,obj];
	},
	
	_translateActivity: function($n,tid,res) {
		var obj;
		
		if( $n.is(':node(subProcess)') ){
			tid = 'process_'+tid;
			obj = res.activities[tid] = this.translateFragment($n);
			obj.markers = obj.markers || [];
			obj.markers.push("subprocess");
		} else {
			tid = 'task_'+tid;
			obj = res.activities[tid] = {};
			
			if( $n.is(':node(sendTask)') ) {
				obj.type = "send";
			} else if ( $n.is(':node(userTask)') ) {
				obj.type = "user";
			} else if ( $n.is(':node(serviceTask)') ) {
				obj.type = "service";
			} else if ( $n.is(':node(receiveTask)') ) {
				obj.type = "receive";
			} else if ( $n.is(':node(manualTask)') ) {
				obj.type = "manual";
			} else if ( $n.is(':node(businessTask)') ) {
				obj.type = "business";
			} else if ( $n.is(':node(scriptTask)') ) {
				obj.type = "script";
			}
		}
		
		if( $n.attr('isForCompensation') === "true" ) {
			obj.markers = obj.markers || [];
			obj.markers.push("compensation");
		}
		if( $n.children(':node(multiInstanceLoopCharacteristics)').length ) {
			obj.markers = obj.markers || [];
			if( $n.children(':node(multiInstanceLoopCharacteristics)').attr('isSequential') === "true" ) {
				obj.markers.push("sequential");
			} else {
				obj.markers.push("parallel");
			}
		}
		if( $n.children(':node(standardLoopCharacteristics)').length ) {
			obj.markers = obj.markers || [];
			obj.markers.push("loop");
		}
		
		if( $n.attr('triggeredByEvent') === "true" ) {
			obj.activityType = "event";
		} else if( $n.is(':node(callActivity)') ) {
			obj.activityType = "call";
		}
		
		return [tid,obj];
	},
	
	_translateGateway: function($n,tid,res) {
		var obj;
		
		tid = 'gw_'+tid;
		obj = res.gateways[tid] = {};
		
		if( $n.is(':node(parallelGateway)') ) {
			obj.type = "parallel";
		} else if( $n.is(':node(eventBasedGateway)') ) {
			obj.type = "event";
		} else if( $n.is(':node(inclusiveGateway)') ) {
			obj.type = "inclusive";
		}
		
		return [tid,obj];
	},
	
	/**
	 * Recursively translates XML to SimpleBPMN notation,
	 * represented as JavaScript Object at this point.
	 *
	 * @param jQuery $root jQuery-ized XML fragment
	 *
	 * @name Converter.translateFragment
	 * @type Object
	 */
	translateFragment: function($root) {
		var self = this,
			idMap = {},
			res = {
				events: {
					start: {},
					intermediate: {},
					end: {}
				},
				activities: {},
				gateways: {},
				flow: {}
			};
			
		$root.children().not(':node(sequenceflow)').not(':node(association)').each(function(){
			var $n = $(this),
				id = $n.attr('id') || '',
				r, tid, obj;
			
			tid = self._translateID(id);
			
			/* Events */
			if( $n.is(':node(startEvent)')
				|| $n.is(':node(endEvent)')
				|| $n.is(':node(boundaryEvent)')
				|| $n.is(':node(intermediateThrowEvent)')
				|| $n.is(':node(intermediateCatchEvent)')
			) {
				r = self._translateEvent($n,tid,res);
			}
			/* Activities */
			else if( $n.is(':node(task)')
				|| $n.is(':node(sendTask)')
				|| $n.is(':node(userTask)')
				|| $n.is(':node(serviceTask)')
				|| $n.is(':node(receiveTask)')
				|| $n.is(':node(manualTask)')
				|| $n.is(':node(businessTask)')
				|| $n.is(':node(scriptTask)')
				|| $n.is(':node(subProcess)')
				|| $n.is(':node(callActivity)')
			) {
				r = self._translateActivity($n,tid,res);
			}
			/* Gateways */
			else if( $n.is(':node(exclusiveGateway)')
				|| $n.is(':node(parallelGateway)')
				|| $n.is(':node(eventBasedGateway)')
				|| $n.is(':node(inclusiveGateway)')
			) {
				r = self._translateGateway($n,tid,res);
			}
			
			if( r ) {
				obj = r[1];
			
				/* Common attributes */
				obj.name = $n.attr('name');
			
				/* Remember mapping between IDs */
				idMap[id] = r[0];
			}
		});
		/* Flow */
		$root.children(':node(sequenceflow), :node(association)').each(function(){
			var $n = $(this),
				id = $n.attr('id') || '',
				flow = [],
				tid;
				
			tid = 'f' + self._translateID(id);
			
			flow[0] = idMap[ $n.attr('sourceRef') ];
			flow[1] = idMap[ $n.attr('targetRef') ];
			
			if( flow[0] && flow[1] ) {
				if( $n.attr('name') ) {
					flow[2] = $n.attr('name');
				}
			
				res.flow[tid] = flow;
			}
		});
		/* Boundary events */
		var e,bid;
		if( res.events && res.events.intermediate ){
			for( e in res.events.intermediate ) {
				if( res.events.intermediate[e].boundaryFor ) {
					bid = idMap[ res.events.intermediate[e].boundaryFor ];
					
					res.activities[bid].boundary = res.activities[bid].boundary || [];
					res.activities[bid].boundary.push(e);
					
					delete res.events.intermediate[e].boundaryFor;
				}
			}
		}
		return res;
	},
	
	/**
	 * Recursively cleans result object - removes empty parts and minimizes it,
	 * when only 'name' attribute is present. Clining is done IN PLACE.
	 *
	 * @param Object results
	 * 
	 * @name Converter.cleanFragment
	 * @type Object
	 */
	cleanFragment: function(results){
		var k, t;
		
		for(k in results){
			if( results[k].name !== undefined ) {
				t = $.extend({},results[k]);
				delete t.name;
				if( $.isEmptyObject(t) ) {
					results[k] = results[k].name;
				}
			}
			if( $.isPlainObject(results[k]) ) {
				results[k] = this.cleanFragment(results[k]);
				
				if( $.isEmptyObject(results[k]) ) {
					delete results[k];
				} else if ( results[k].name === "" ) {
					delete results[k].name;
				}
			}
		}
		
		return results;
	},
	
	/**
	 * Cleans and formats the results.
	 *
	 * @param Object results
	 *
	 * @name Converter.processResults
	 * @type String
	 */
	processResults: function(results){
		results = this.cleanFragment(results);
		return JSON.stringify(results, null, '\t');
	}
};
